--[[

applets.AlsaBufferSize.AlsaBufferSizeApplet - ALSA buffer adjustment

This applet was created to allow the ALSA buffers set up by jive_alsa
to be changed. It is specific to the Squeezebox Radio.

The motivation for this is the occurrence of 'Bass drop-out' that has
been reported by some users under the 'Bass amp problem' thread on the
Logitech Squeezebox user forums. The occasions of this effect seem to be
somewhat sporadic, and probably dependent on the particular stream being
played.

Investigation suggests that the problem may be triggered by ALSA
under-runs, that these under-runs are most likely to occur when a 48kHz
sample rate stream is being played, and, in particular but not
exclusively, when an effects sound is being generated. Under these
circumstance jive_alsa will be more loaded than usual.

In tests, it has been found that the problem appears to be eliminated
if the ALSA buffer size is increased from the system default of 20
milliseconds to 30 milliseconds - i.e. three periods of 10 milliseconds
instead of two periods.

This probably works because the addition of a third period allows
jive_alsa a little more head-room. If it should fail to generate a chunk
of samples within the 10 ms allowed, there will be an additional 10 ms
of prepared samples awaiting play-back by ALSA. So ALSA under-runs
appear to be eliminated.


Copyright (c) 2019 Martin Williams, mrwlists@gmail.com

Released under the BSD license for use with the Logitech Squeezeplay
application.

Version 1.01 - released 15 Feb 2019

--]]


-- stuff we use
local ipairs, table, tostring, type = ipairs, table, tostring, type

local oo            = require("loop.simple")
local Applet        = require("jive.Applet")

local RadioButton   = require("jive.ui.RadioButton")
local RadioGroup    = require("jive.ui.RadioGroup")
local Window        = require("jive.ui.Window")
local Textarea      = require('jive.ui.Textarea')
local SimpleMenu    = require("jive.ui.SimpleMenu")
local Label         = require("jive.ui.Label")
local Popup         = require("jive.ui.Popup")
local Icon          = require("jive.ui.Icon")
local Group         = require("jive.ui.Group")
local Task          = require("jive.ui.Task")
local Framework     = require("jive.ui.Framework")
local EVENT_UNUSED  = jive.ui.EVENT_UNUSED
local EVENT_CONSUME = jive.ui.EVENT_CONSUME

local appletManager	= appletManager

module(...)
 oo.class(_M, Applet)



function menu(self, menuItem)

	-- Declared at the start so as to be available to item callbacks.
	local startWindow = Window("window", menuItem.text)
	-- Disable screensaver & title bar right hand button (NP) icon, I
	-- don't like these on a settings menu.
	-- In reality the button is not in point for a baby, it doesn't
	-- have a touch display.
	startWindow:setAllowScreensaver(false)
	startWindow:setButtonAction("rbutton", nil)

	-- Where the work will be done.
	local mainMenu = self:buildMainMenu(menuItem)

	-- Build the preliminary menu.
	local startMenu = SimpleMenu("menu")
	startMenu:setHeaderWidget(Textarea("help_text", self:string("START_MENU_HELP")))

	-- This item will transition to the main menu, replacing this
	-- preliminary window/menu.
	startMenu:addItem ({
		text     = self:string("CONTINUE_ITEM"),
		sound    = "WINDOWSHOW",
		callback = function (event, menuItem)
			-- Replaces our start menu with the main menu

			-- Use the same title as our start window
			local window = Window("window", startWindow:getTitle())
			-- No screensaver & title bar right hand button (NP) icon.
			window:setAllowScreensaver(false)
			window:setButtonAction("rbutton", nil)
			window:addWidget(mainMenu)

			--Transition to our new window, replacing the start window.
			self:tieWindow(window)
			window:replace(startWindow)

			-- Stash these for use by doRestart.
			self.menu   = mainMenu
			self.window = window
		end
	})

	-- The explanatory note.
	startMenu:addItem ({
		text     = self:string("EXPLANATION_TITLE"),
		sound    = "WINDOWSHOW",
		callback = function (event, menuItem)
			local window = Window("text_list", self:string("EXPLANATION_TITLE"))
			-- 'help_text' gives a better display than 'text' or 'multiline_text'
			local text   = Textarea('help_text', self:string("EXPLANATORY_TEXT"))
			window:addWidget(text)
			self:tieAndShowWindow(window)
		end
	})

	-- Finally, show the window.
	startWindow:addWidget(startMenu)
	self:tieAndShowWindow(startWindow)

end

function buildMainMenu(self, menuItem)

	-- The applet meta has provided four parameters giving the current
	-- and system default values of alsaPlaybackBufferTime and
	-- alsaPlaybackPeriodCount.

	local defBufTime  = menuItem.defaultBufferTime
	local defPerCount = menuItem.defaultPeriodCount
	local curBufTime  = menuItem.startupBufferTime
	local curPerCount = menuItem.startupPeriodCount


	local menu = SimpleMenu("menu")
	menu:setHeaderWidget(Textarea("help_text", self:string("MAIN_MENU_HELP")))

	-- First menu item gives reboot option.
	-- Baby always has this, nevertheless doRestart will cater for
	-- circumstances where reboot is not available.

--	if appletManager:hasService("reboot") then
		menu:addItem ({
			suppressPlay = true, -- Used by menu's action listener.
			text         = self:string("RESTART_SYSTEM_ITEM"),
			sound        = "SELECT",
			callback     = function (event, menuItem)
				self:doRestart()
			end
		})

		-- Radio buttons all respond to the 'Play' action. Left alone,
		-- the 'Restart system' menu item will not - it results in music.
		-- Inconsistent, so catch the 'Play' action on 'Restart system'.
		menu:addActionListener("play", menu,
			function (menu)
				local menuitem = menu.items[menu:getSelectedIndex()]
				if menuitem.suppressPlay then
					Framework:bump()
					return EVENT_CONSUME
				else
					return EVENT_UNUSED
				end
			end
		)
--	end


	-- Next menu items are a set of RadioButton choices for the ALSA
	-- buffer size.

	-- From 2 up to 5 ALSA periods of 10 milliseconds each.
	-- Hmmm, in practice we only need the 20 & 30 ms options...
	local options = {
			{ desc = "20 ms", time = 20000, count = 2 },
			{ desc = "30 ms", time = 30000, count = 3 },
			{ desc = "40 ms", time = 40000, count = 4 },
			{ desc = "50 ms", time = 50000, count = 5 },
	}
	-- Highlight which option is currently active and which is the
	-- system default.
	local defaultSeen = false
	for _, opt in ipairs(options) do
		if opt.time == curBufTime and opt.count == curPerCount then
			opt.desc = opt.desc .. " - " ..
			                tostring(self:string("ACTIVE_TXT"))
		end
		if opt.time == defBufTime and opt.count == defPerCount then
			opt.desc = opt.desc .. " - " ..
			                tostring(self:string("SYS_DEFAULT_TXT"))
			defaultSeen = true
		end
	end
	if not defaultSeen then
		-- Won't happen unless omitted from options table. In which case,
		-- add a system default option as the first item.
		table.insert(options, 1,
			{ desc  = defBufTime / 1000 .. " ms - " ..
			               tostring(self:string("SYS_DEFAULT_TXT")),
			  time  = defBufTime,
			  count = defPerCount
			}
		)
		log:debug("Added jive_alsa defaults: ", defBufTime, " ", defPerCount)
	end

	-- Add each option to the menu.

	-- We've already checked that the SqueezeboxBaby applet and its
	-- settings are accessible in the applet meta, so no pcall needed.
	local baby         = appletManager:getAppletInstance("SqueezeboxBaby")
	local babySettings = baby:getSettings()
	local group        = RadioGroup()
	for _, opt in ipairs(options) do
		menu:addItem({
			text  = opt.desc,
			style = 'item_choice',
			check = RadioButton(
				"radio", group,
				function(event, menuItem)
					-- Set these up ready for next restart
					log:info("Setting ALSA buffer time and period count to: ",
					              opt.time, " : ", opt.count)
					babySettings.alsaPlaybackBufferTime  = opt.time
					babySettings.alsaPlaybackPeriodCount = opt.count
					-- No need to explicitly store settings for Baby, it
					-- does it automatically on a normal reboot.
					-- baby:storeSettings()
				end,
				-- Energize radio button if it matches our chosen value.
				babySettings.alsaPlaybackBufferTime == opt.time
					and babySettings.alsaPlaybackPeriodCount == opt.count
			)
		})
	end

	-- Stash SqueezeboxBaby instance for use by doRestart.
	self.baby = baby

	return menu

end


-- Reboot system if available, else display a restart message.
-- Restart message expects current window to be displaying a menu,
-- which will be replaced.

-- Requires that self.window and self.menu refer to the current
-- window/menu and self.baby refers to the instance of the
-- SqueezeboxBaby applet.

function doRestart(self)

	if not appletManager:hasService("reboot") then
		-- Shouldn't ever happen on a Baby
		log:warn("No reboot service available")
		self.window:removeWidget(self.menu)
		self.window:addWidget(Textarea("help_text", self:string("RESTART_APP_TEXT")))
		-- The Baby's reboot service would have done this...
		self.baby:storeSettings()

	else
		-- Essentials pinched from SetupAppletInstaller
		self.task = Task("restarter", self, function()
			-- Generate animated restarting screen
			local icon          = Icon("icon_connecting")
			local animatelabel  = Label("text", self:string("RESTART_JIVE_LABEL"))
			local animatewindow = Popup("waiting_popup")
			animatewindow:addWidget(icon)
			animatewindow:addWidget(animatelabel)
			animatewindow:show()
			-- Wait for a few seconds before rebooting
			local t = Framework:getTicks()
			while (t + 2000) > Framework:getTicks() do
				Task:yield(true)
			end
			log:info("RESTARTING JIVE...")
			appletManager:callService("reboot")
			-- This should never happen.
			log:warn("REBOOT SEEMS TO HAVE FAILED")
			-- The Baby's reboot service would have done this...
			self.baby:storeSettings()
			animatewindow:hide()
			-- Display an instructional popup
			self:showPopup(self:string("RESTART_JIVE_FAILED"), 5000)
		end)

		self.task:addTask()
		log:debug("Task added")
	end
end


-- A simple popup.

function showPopup(self, message, msecs)

		-- Default to 3 seconds, but never less than 1 second.
		if type(msecs) ~= "number" or msecs < 1000 then
			msecs = 3000
		end
		-- Create a Popup object, with 'toast_popup_text' skin style
		local popupWindow = Popup('toast_popup_text')
		popupWindow:setAlwaysOnTop(true)

		local popupMsg = Group("group", {
			text = Textarea('toast_popup_textarea',message),
		})
		popupWindow:addWidget(popupMsg)
		-- Display for msecs / 1000 seconds
		popupWindow:showBriefly(msecs, nil, Window.transitionPushPopupUp, Window.transitionPushPopupDown)
 end



