--[[

applets.AlsaBufferSize.AlsaBufferSizeMeta - ALSA buffer adjustment

Copyright (c) 2019 Martin Williams, mrwlists@gmail.com

Released under the BSD license for use with the Logitech Squeezeplay
application.

Version 1.0 - released 12 Feb 2019

--]]



local oo            = require("loop.simple")
local AppletMeta    = require("jive.AppletMeta")
local appletManager = appletManager
local jiveMain      = jiveMain

local assert, error, pcall, type = assert, error, pcall, type

module(...)
oo.class(_M, AppletMeta)

function jiveVersion(meta)
	return 1, 1
end


--[[
Firmware versions seen:

http://update.slimdevices.com/update/firmware/
/etc/version

7.4.0 r7790 Mon Sep 28 04:06:53 PDT 2009
7.4.1 r7915 Wed Oct 21 07:53:08 PDT 2009
7.4.2 r8423 Fri Jan 29 06:47:49 MST 2010
7.4.3 r8553 Tue Feb 23 05:57:52 MST 2010
7.5.0 r8673 Thu Apr 1 08:11:07 MDT 2010
7.5.1 r9218 Mon Nov 8 13:47:43 MST 2010
7.5.3 r9377 Thu Mar 17 13:52:38 MDT 2011
7.5.4 r9408 Wed Apr 6 11:44:27 MDT 2011
7.6.0 r9458 Mon Jul 25 12:00:23 MDT 2011
7.6.1 r9482 Mon Aug 15 08:13:11 MDT 2011
7.6.2 r9545 Wed Sep 28 07:34:39 MDT 2011
7.7.0 r9546 Wed Sep 28 09:38:39 MDT 2011
7.7.1 r9557 Mon Nov 28 12:03:56 MST 2011
7.7.2 r9663 Mon Mar 19 10:17:11 MDT 2012
7.7.3 r16676 Fri Feb 14 09:25:26 PST 2014
Base build revision:  bad080aecfec8226a4c1699b29d32cbba4ba396b

And there is also:

baby_10.0.0_r16943.bin 24-Aug-2012 04:35

--]]

--[[
local JIVE_VERSION = jive.JIVE_VERSION
Is how we could check the version if required.
--]]

function registerApplet(meta)

	-- Early (pre-release) versions of Squeezeplay/Baby handled the
	-- launch of jive_alsa by way of the 'Playback' applet, and startup
	-- parameters were held in that Applet.
	-- Released versions handled matters in the SqueezeboxBaby applet.
	-- This was consequent on the adoption of an RT kernel, and a change
	-- in the default buffer size from 30ms to 20ms.

	-- This Applet does not cater for starting up jive_alsa in any way
	-- other than the method currently in use. We check that our Baby's
	-- settings appear to be consistent with this.

	-- Unlike conventional Applets, SqueezeboxBaby is loaded at meta
	-- evaluation stage. So, provided we are loaded afterwards, we can
	-- detect if we are running on a baby, and verify that its settings
	-- are consistent with expectation. Our 'loadPriority.lua' should be
	-- set appropriately (although it defaults to 100, which is well
	-- after SqueezeBoxBaby).


	local defaultBufferTime, defaultPeriodCount
	local startupBufferTime, startupPeriodCount
	local ok, res = pcall(
		function()
			local baby = assert(appletManager:getAppletInstance("SqueezeboxBaby"),
						"SqueezeboxBaby is not running.")
			defaultBufferTime  = baby:getDefaultSettings()["alsaPlaybackBufferTime"]
			defaultPeriodCount = baby:getDefaultSettings()["alsaPlaybackPeriodCount"]
			startupBufferTime  = baby:getSettings()["alsaPlaybackBufferTime"]
			startupPeriodCount = baby:getSettings()["alsaPlaybackPeriodCount"]

			-- Check that the default values retrieved are as we expect.

			-- Current default is 20000, but has been 30000. Reject anything else.
			assert(defaultBufferTime == 20000 or defaultBufferTime == 30000,
						"Expecting default buffer size of 20000 or 30000")
			assert(type(defaultPeriodCount) == "number",
						"Default period count is not numeric")
			assert(defaultBufferTime / defaultPeriodCount == 10000,
						"Expecting default period size of 10000")

			-- Check that the start up values are as we might expect.

			-- Stock jive and this applet always set numeric buffer time.
			assert(type(startupBufferTime) == "number", "Buffer size is not numeric")
			-- Stock jive and this applet always set numeric period count.
			assert(type(startupPeriodCount) == "number", "Period count is not numeric")
			-- Stock jive and this applet always set a period size of 10000.
			-- On second thoughts, allow this in case user has done something curious.
--			assert(startupBufferTime / startupPeriodCount == 10000, "Expecting period size of 10000")

		end
	)

	if not ok then
		log:error("The SqueezeboxBaby Applet is not behaving as expected. Abandoning...\n", res)
		return
	end


	jiveMain:addItem(
		meta:menuItem(
			"alsaBufferSizeApplet",
			"advancedSettings",
			"MAIN_MENU_TITLE",
			function(applet, ...) applet:menu(...) end,
			20, -- Weight reasonably close to the top
			-- We pass jive_alsa's start up parameters through to our
			-- applet, so we know how jive_alsa has been launched.
			-- And we might as well pass the system defaults, too.
			{
				startupBufferTime  = startupBufferTime,
				startupPeriodCount = startupPeriodCount,
				defaultBufferTime  = defaultBufferTime,
				defaultPeriodCount = defaultPeriodCount
			},
			"hm_advancedSettings"
		)
	)
end
